//$Id: CRect.cpp,v 1.7 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CRect.hpp"
#include "../Basics/CStringTokeniser.hpp"
#include "CMathTools.hpp"
using Exponent::MathTools::CRect;
using Exponent::MathTools::CMathTools;
using Exponent::Basics::CStringTokeniser;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CRect, CCountedObject);

//	===========================================================================
CRect::CRect(const long left, const long top, const long width, const long height) 
	 : m_width(0)
	 , m_height(0)
	 , m_left(0)
	 , m_right(0)
	 , m_top(0)
	 , m_bottom(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CRect);
	this->setRect(left, top, width, height);
}

//	===========================================================================
CRect::CRect(const CRect &rect)
	 : m_width(0)
	 , m_height(0)
	 , m_left(0)
	 , m_right(0)
	 , m_top(0)
	 , m_bottom(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CRect);
	this->setRect(rect.getLeft(), rect.getTop(), rect.getWidth(), rect.getHeight());
}

//	===========================================================================
CRect::CRect(const CPoint &origin, const CDimension &dimension)
	 : m_width(0)
	 , m_height(0)
	 , m_left(0)
	 , m_right(0)
	 , m_top(0)
	 , m_bottom(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CRect);
	this->setRect(origin.getXPosition(), origin.getYPosition(), dimension.getWidth(), dimension.getHeight());
}

//	===========================================================================
CRect::~CRect()
{
	EXPONENT_CLASS_DESTRUCTION(CRect);
}

//	===========================================================================
CRect &CRect::operator = (const CRect &rect)
{
	this->setRect(rect.getLeft(), rect.getTop(), rect.getWidth(), rect.getHeight());
	return (*this);
}

//	===========================================================================
bool CRect::operator == (const CRect &rect)
{
	return (m_left   == rect.getLeft()  &&
		    m_top    == rect.getTop()   &&
		    m_width  == rect.getWidth() &&
		    m_height == rect.getHeight());
}

//	===========================================================================
void CRect::setRect(const long left, const long top, const long width, const long height)
{
	m_left = left;
	m_right = left + width;
	m_top = top;
	m_bottom = top + height;
	m_height = height;
	m_width = width;
}

//	===========================================================================
void CRect::setRect(const CPoint &origin, const CDimension &dimension)
{
	this->setRect(origin.getXPosition(), origin.getYPosition(), dimension.getWidth(), dimension.getHeight());
}

//	===========================================================================
void CRect::setLeft(const long left)
{
	m_left = left;
	m_right = left + m_width;
}

//	===========================================================================
void CRect::setWidth(const long width)
{
	m_width = width;
	m_right = m_left + width;
}

//	===========================================================================
void CRect::setTop(const long top)
{
	m_top = top;
	m_bottom = top + m_height;
}

//	===========================================================================
void CRect::setHeight(const long height)
{
	m_height = height;
	m_bottom = m_top + height;
}

//	===========================================================================
void CRect::setOrigin(const CPoint &point)
{
	m_left   = point.getXPosition();
	m_right  = m_left + m_width;
	m_top    = point.getYPosition();
	m_bottom = m_top + m_height;
}

//	===========================================================================
void CRect::setDimensions(const CDimension &dimension)
{
	this->setWidth(dimension.getWidth());
	this->setHeight(dimension.getHeight());
}

//	===========================================================================
void CRect::offset(const CPoint &point)
{
	this->setLeft(m_left + point.getXPosition());
	this->setTop( m_top  + point.getYPosition());
}

//	===========================================================================
void CRect::inset(const long amount)
{
	m_left += amount;
	m_top  += amount;

	m_right  -= amount;
	m_bottom -= amount;

	m_width  -= amount * 2;
	m_height -= amount * 2;
}

//	===========================================================================
void CRect::scale(const double amount)
{
	this->setLeft((long)(m_left * amount));
	this->setTop((long)(m_top * amount));
	this->setWidth((long)(m_width * amount));
	this->setHeight((long)(m_height * amount));
}

//	===========================================================================
CRect *CRect::getIntersectionArea(const CRect &rect1, const CRect &rect2)
{
	CRect *theRect = new CRect();

	theRect->setLeft(CMathTools::maximum(rect1.getLeft(), rect2.getLeft()));
	theRect->setTop(CMathTools::maximum(rect1.getTop(), rect2.getTop()));
	theRect->setWidth(CMathTools::minimum(rect1.getRight(), rect2.getRight()) - theRect->getLeft());
	theRect->setHeight(CMathTools::minimum(rect1.getBottom(), rect2.getBottom()) - theRect->getTop());

	if (theRect->getWidth() < 0 || theRect->getHeight() < 0)
	{
		theRect->setRect(0, 0, 0, 0);
	}

	return theRect;
}

//	===========================================================================
void CRect::getIntersectionArea(const CRect &rect1, const CRect &rect2, CRect &output)
{
	output.setRect(CMathTools::maximum(rect1.getLeft(), rect2.getLeft()),
				   CMathTools::maximum(rect1.getTop(), rect2.getTop()),
				   CMathTools::minimum(rect1.getRight(), rect2.getRight())   - CMathTools::maximum(rect1.getLeft(), rect2.getLeft()),
				   CMathTools::minimum(rect1.getBottom(), rect2.getBottom()) - CMathTools::maximum(rect1.getTop(), rect2.getTop()));
}

//	===========================================================================
void CRect::unionWith(const CRect &rect)
{
	// Create the union rectangle
	CRect theResult;
	theResult.m_left   = CMathTools::minimum(m_left, rect.m_left);
	theResult.m_top    = CMathTools::minimum(m_top,  rect.m_top);
	theResult.m_width  = CMathTools::maximum(m_right, rect.m_right - theResult.m_left);
	theResult.m_height = CMathTools::maximum(m_bottom, rect.m_bottom - theResult.m_top);

	// Now set ourselves to this rect
	this->setRect(theResult.m_left, theResult.m_top, theResult.m_width, theResult.m_height);
}

//	===========================================================================
void CRect::getObjectDescription(char *string, const long size) const
{
	CString description;
	description.setStringWithFormat("[%li,%li,%li,%li]", this->getLeft(), this->getTop(), this->getWidth(), this->getHeight());
	description.getString(string, size);
}

//	===========================================================================
long CRect::getCentralXPosition() const
{
	return this->getLeft() + (this->getWidth() / 2);
}

//	===========================================================================
long CRect::getCentralYPosition() const
{
	return this->getTop() + (this->getHeight() / 2);
}

#ifdef WIN32

//	===========================================================================
RECT CRect::getAsRect()
{
	RECT rect;
	rect.left   = m_left;
	rect.right  = m_right;
	rect.top    = m_top;
	rect.bottom = m_bottom;
	return rect;
}

//	===========================================================================
void CRect::setFromRect(RECT &rect)
{
	m_left   = rect.left;
	m_right  = rect.right;
	m_top    = rect.top;
	m_bottom = rect.bottom;
	m_width  = m_right - m_left;
	m_height = m_bottom - m_top;
}

#else

//	===========================================================================
Rect *CRect::getAsRect() const
{
	Rect *rect   = new Rect;

	rect->left   = m_left;
	rect->top    = m_top;
	rect->bottom = m_bottom;
	rect->right  = m_right;

	return rect;
}

//	===========================================================================
void CRect::setFromRect(Rect &rect)
{
	m_left   = rect.left;
	m_right  = rect.right;
	m_top    = rect.top;
	m_bottom = rect.bottom;
	m_width  = m_right - m_left;
	m_height = m_bottom - m_top;
}
#endif

//	===========================================================================
CRect CRect::convertFromString(const CString &string)
{
	// Tokenise it
	CStringTokeniser tokeniser(string, ',');

	// Check its the correct number of tokens
	if (tokeniser.getNumberOfTokens() != 4)
	{
		return CRect(0, 0, 0, 0);
	}

	// The buffer
	CString buffer;
	long rect[4] = { 0, 0, 0, 0 };

	for (long i = 0; i < 4; i++)
	{
		// Get each token
		if (!tokeniser.getNextToken(buffer))
		{
			return CRect(0, 0, 0, 0);
		}
		else
		{
			rect[i] = CString::toLong(buffer.getString());
		}
	}

	// Return the rect
	return CRect(rect[0], rect[1], rect[2], rect[3]);
}